package com.monitor.parentalcontrol;

import android.Manifest;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.pm.PackageManager;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.os.PowerManager;
import android.provider.Settings;
import android.view.View;
import android.widget.Button;
import android.widget.EditText;
import android.widget.Toast;

import androidx.appcompat.app.AppCompatActivity;
import androidx.core.app.ActivityCompat;
import androidx.core.content.ContextCompat;

import com.android.volley.Request;
import com.android.volley.RequestQueue;
import com.android.volley.Response;
import com.android.volley.VolleyError;
import com.android.volley.toolbox.StringRequest;
import com.android.volley.toolbox.Volley;

import org.json.JSONObject;

import java.util.HashMap;
import java.util.Map;

public class MainActivity extends AppCompatActivity {

    private static final int PERMISSION_REQUEST_CODE = 100;
    private EditText etChildName, etParentName, etParentEmail;
    private Button btnRegister;
    private SharedPreferences sharedPreferences;
    private String API_URL = "https://myapplicationapp.infinityfreeapp.com/api/";

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        
        sharedPreferences = getSharedPreferences("MonitorPrefs", MODE_PRIVATE);
        
        // Check if already registered
        if (sharedPreferences.getBoolean("isRegistered", false)) {
            startMonitoringService();
            finish();
            return;
        }

        setContentView(R.layout.activity_main);

        etChildName = findViewById(R.id.etChildName);
        etParentName = findViewById(R.id.etParentName);
        etParentEmail = findViewById(R.id.etParentEmail);
        btnRegister = findViewById(R.id.btnRegister);

        // Request permissions first
        requestAllPermissions();

        btnRegister.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (checkAllPermissions()) {
                    registerUser();
                } else {
                    Toast.makeText(MainActivity.this, 
                        "সব পারমিশন দিন", Toast.LENGTH_SHORT).show();
                    requestAllPermissions();
                }
            }
        });
    }

    private void requestAllPermissions() {
        // Request battery optimization disable
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            Intent intent = new Intent();
            String packageName = getPackageName();
            PowerManager pm = (PowerManager) getSystemService(POWER_SERVICE);
            if (!pm.isIgnoringBatteryOptimizations(packageName)) {
                intent.setAction(Settings.ACTION_REQUEST_IGNORE_BATTERY_OPTIMIZATIONS);
                intent.setData(Uri.parse("package:" + packageName));
                startActivity(intent);
            }
        }

        // Request runtime permissions
        String[] permissions = {
            Manifest.permission.READ_SMS,
            Manifest.permission.RECEIVE_SMS,
            Manifest.permission.READ_CALL_LOG,
            Manifest.permission.READ_CONTACTS,
            Manifest.permission.READ_PHONE_STATE
        };

        ActivityCompat.requestPermissions(this, permissions, PERMISSION_REQUEST_CODE);
    }

    private boolean checkAllPermissions() {
        String[] permissions = {
            Manifest.permission.READ_SMS,
            Manifest.permission.RECEIVE_SMS,
            Manifest.permission.READ_CALL_LOG,
            Manifest.permission.READ_CONTACTS,
            Manifest.permission.READ_PHONE_STATE
        };

        for (String permission : permissions) {
            if (ContextCompat.checkSelfPermission(this, permission) 
                != PackageManager.PERMISSION_GRANTED) {
                return false;
            }
        }
        return true;
    }

    private void registerUser() {
        String childName = etChildName.getText().toString().trim();
        String parentName = etParentName.getText().toString().trim();
        String parentEmail = etParentEmail.getText().toString().trim();

        if (childName.isEmpty() || parentName.isEmpty() || parentEmail.isEmpty()) {
            Toast.makeText(this, "সব তথ্য পূরণ করুন", Toast.LENGTH_SHORT).show();
            return;
        }

        String deviceId = Settings.Secure.getString(getContentResolver(), 
            Settings.Secure.ANDROID_ID);

        // Send registration to server
        String url = API_URL + "register.php";

        RequestQueue queue = Volley.newRequestQueue(this);
        StringRequest request = new StringRequest(Request.Method.POST, url,
            new Response.Listener<String>() {
                @Override
                public void onResponse(String response) {
                    try {
                        JSONObject jsonResponse = new JSONObject(response);
                        if (jsonResponse.getBoolean("success")) {
                            Toast.makeText(MainActivity.this, 
                                "রেজিস্ট্রেশন সফল!", Toast.LENGTH_SHORT).show();
                            
                            // Save registration data
                            SharedPreferences.Editor editor = sharedPreferences.edit();
                            editor.putBoolean("isRegistered", true);
                            editor.putString("deviceId", deviceId);
                            editor.putString("childName", childName);
                            editor.putString("parentEmail", parentEmail);
                            editor.apply();

                            // Start monitoring service
                            startMonitoringService();
                            finish();
                        } else {
                            Toast.makeText(MainActivity.this, 
                                "রেজিস্ট্রেশন ব্যর্থ: " + jsonResponse.getString("message"), 
                                Toast.LENGTH_SHORT).show();
                        }
                    } catch (Exception e) {
                        Toast.makeText(MainActivity.this, 
                            "Error: " + e.getMessage(), Toast.LENGTH_SHORT).show();
                    }
                }
            },
            new Response.ErrorListener() {
                @Override
                public void onErrorResponse(VolleyError error) {
                    Toast.makeText(MainActivity.this, 
                        "সার্ভার সংযোগ ব্যর্থ!", Toast.LENGTH_SHORT).show();
                }
            }) {
            @Override
            protected Map<String, String> getParams() {
                Map<String, String> params = new HashMap<>();
                params.put("device_id", deviceId);
                params.put("child_name", childName);
                params.put("parent_name", parentName);
                params.put("parent_email", parentEmail);
                return params;
            }
        };

        queue.add(request);
    }

    private void startMonitoringService() {
        Intent serviceIntent = new Intent(this, MonitoringService.class);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            startForegroundService(serviceIntent);
        } else {
            startService(serviceIntent);
        }
    }

    @Override
    public void onRequestPermissionsResult(int requestCode, String[] permissions, int[] grantResults) {
        super.onRequestPermissionsResult(requestCode, permissions, grantResults);
        if (requestCode == PERMISSION_REQUEST_CODE) {
            boolean allGranted = true;
            for (int result : grantResults) {
                if (result != PackageManager.PERMISSION_GRANTED) {
                    allGranted = false;
                    break;
                }
            }
            if (!allGranted) {
                Toast.makeText(this, "অ্যাপ চালাতে সব পারমিশন দরকার", Toast.LENGTH_LONG).show();
            }
        }
    }
}
